<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Package;
use App\Models\UserPackage;
use App\Models\Spin;
use App\Models\UserSpin;
use App\Models\User;
use App\Models\UserLedger;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PackageController extends Controller
{
    // Show all active packages
    public function index()
    {
        $packages = Package::where('status', 1)->get();
        return view('user.packages.index', compact('packages'));
    }

    // Buy package
    public function buy($id)
    {
        $user = auth()->user();
        // 🔥 AUTO-EXPIRE OLD PACKAGES
    UserPackage::where('user_id', $user->id)
        ->where('status', 'active')
        ->whereNotNull('end_date')
        ->where('end_date', '<', now())
        ->update(['status' => 'expired']);

        $package = Package::findOrFail($id);

        // --- Free package logic ---
        if ($package->type === 'free') {

    // ❌ Block if user EVER used free plan
    $alreadyBought = UserPackage::where('user_id', $user->id)
        ->where('package_id', $package->id)
        ->exists();

    if ($alreadyBought) {
        return back()->with('error', 'Free plan can only be joined once.');
    }

    UserPackage::create([
        'user_id' => $user->id,
        'package_id' => $package->id,
        'start_date' => now(),
        'end_date' => now()->addDays($package->validity_days),
        'status' => 'active',
    ]);

    $this->assignSpins($user->id, $package);

    return back()->with('success', 'Free plan activated.');
}


        // --- Paid package logic ---
        // --- Paid package logic ---
// --- Paid package logic ---
$activePackage = UserPackage::where('user_id', $user->id)
    ->where('status', 'active')
    ->whereHas('package', function ($q) {
        $q->where('type', 'paid');
    })
    ->with('package')
    ->first();

// Prevent downgrade
if ($activePackage && $package->price <= $activePackage->package->price) {
    return back()->with('error', 'You can only upgrade to a higher job level.');
}

// Define refundAmount OUTSIDE
$refundAmount = 0;

// Balance check (refund applies first)
if ($user->balance < $package->price) {
    return back()->with('error', 'Balance Low. Please Recharge.');
}

DB::transaction(function () use ($user, $package, $activePackage, &$refundAmount) {


// 🔥 EXPIRE FREE PLAN WHEN UPGRADING
UserPackage::where('user_id', $user->id)
    ->whereHas('package', function ($q) {
        $q->where('type', 'free');
    })
    ->where('status', 'active')
    ->update(['status' => 'expired']);



    // PARTIAL REFUND
    if ($activePackage) {
        $totalDays = $activePackage->package->validity_days;
        $usedDays = $activePackage->start_date->diffInDays(now());
        $remainingDays = max($totalDays - $usedDays, 0);

        if ($totalDays > 0 && $remainingDays > 0) {
            $refundAmount = round(
                ($activePackage->package->price / $totalDays) * $remainingDays,
                2
            );

            $user->increment('balance', $refundAmount);

            UserLedger::create([
                'user_id' => $user->id,
                'reason' => 'package_upgrade_partial_refund',
                'perticulation' => 'Partial refund for package upgrade',
                'amount' => $refundAmount,
                'credit' => $refundAmount,
                'status' => 'approved',
                'date' => now(),
            ]);
        }

        $activePackage->update(['status' => 'expired']);
    }

    // Deduct new package price
    $user->decrement('balance', $package->price);

    // Assign new package
    UserPackage::create([
        'user_id' => $user->id,
        'package_id' => $package->id,
        'start_date' => now(),
        'end_date' => $package->validity_days
            ? now()->addDays($package->validity_days)
            : null,
        'status' => 'active',
    ]);

    $this->assignSpins($user->id, $package);
    $user->increment('credit_score', 2);
    $this->distributeReferralCommission($user, $package->price);
});

// SUCCESS MESSAGE (SAFE)
if ($refundAmount > 0) {
    return back()->with(
        'success',
        'Job level upgraded successfully. Refund applied: ' . number_format($refundAmount, 2)
    );
}

return back()->with('success', 'Job secured successfully.');
    }

    // Assign spins to user
    private function assignSpins($userId, $package)
    {
        if ($package->spin_count > 0) {
            $spins = Spin::where('status', 1)->inRandomOrder()->take($package->spin_count)->get();
            foreach ($spins as $spin) {
                UserSpin::create([
                    'user_id' => $userId,
                    'spin_id' => $spin->id,
                    'reward_value' => $spin->reward_value,
                ]);
            }
        }
    }

    // 3-Level referral commission (only for paid packages)
    private function distributeReferralCommission($buyer, $packagePrice)
    {
        if ($packagePrice <= 0) return; // Skip for free packages

        $first_ref = User::where('ref_id', $buyer->ref_by)->first();
        if ($first_ref){
            $amount = $packagePrice * setting('first_level') / 100;
            $first_ref->increment('balance', $amount);
            $first_ref->increment('credit_score', setting('first_level_points', 5));

            $ledger = new UserLedger();
            $ledger->user_id = $first_ref->id;
            $ledger->get_balance_from_user_id = $buyer->id;
            $ledger->reason = 'refer_bonus';
            $ledger->perticulation = 'Congratulations';
            $ledger->amount = $amount;
            $ledger->credit = $amount;
            $ledger->status = 'approved';
            $ledger->step = 'first';
            $ledger->date = now();
            $ledger->save();

            $second_ref = User::where('ref_id', $first_ref->ref_by)->first();
            if ($second_ref){
                $amount = $packagePrice * setting('second_level') / 100;
                $second_ref->increment('balance', $amount);
                $second_ref->increment('credit_score', setting('second_level_points', 5));

                $ledger = new UserLedger();
                $ledger->user_id = $second_ref->id;
                $ledger->get_balance_from_user_id = $buyer->id;
                $ledger->reason = 'refer_bonus';
                $ledger->perticulation = 'Congratulations';
                $ledger->amount = $amount;
                $ledger->credit = $amount;
                $ledger->status = 'approved';
                $ledger->step = 'second';
                $ledger->date = now();
                $ledger->save();

                $third_ref = User::where('ref_id', $second_ref->ref_by)->first();
                if ($third_ref){
                    $amount = $packagePrice * setting('third_level') / 100;
                    $third_ref->increment('balance', $amount);
                    $third_ref->increment('credit_score', setting('third_level_points', 5));

                    $ledger = new UserLedger();
                    $ledger->user_id = $third_ref->id;
                    $ledger->get_balance_from_user_id = $buyer->id;
                    $ledger->reason = 'refer_bonus';
                    $ledger->perticulation = 'Congratulations';
                    $ledger->amount = $amount;
                    $ledger->credit = $amount;
                    $ledger->status = 'approved';
                    $ledger->step = 'third';
                    $ledger->date = now();
                    $ledger->save();
                }
            }
        }
    }
}
