<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\WealthPlan;
use App\Models\UserWealth;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use DB;

class WalletController extends Controller
{
    // Show available plans and user investments
    public function index()
    {
        $user = Auth::user();
        $plans = WealthPlan::where('status', 1)->get(); // active plans only

        // Fetch user investments with plan relation
        $investments = UserWealth::with('plan')
            ->where('user_id', $user->id)
            ->get();

        // Optional: update daily earnings if any pending
        foreach ($investments as $inv) {
            $this->updateDailyEarning($inv, $user);
        }

        return view('user.wealth.index', compact('plans', 'investments', 'user'));
    }

    // Submit new investment
    public function invest(Request $request)
    {
        $request->validate([
            'plan_id' => 'required|exists:wealth_plans,id',
            'amount' => 'required|numeric|min:1'
        ]);

        $user = Auth::user();
        $plan = WealthPlan::findOrFail($request->plan_id);

        if ($request->amount > $user->balance) {
            return back()->with('error', 'Insufficient balance!');
        }

        // Deduct user balance
        $user->balance -= $request->amount;
        $user->save();

        $start = Carbon::now();
        $end = $start->copy()->addDays($plan->days);

        // Create investment record
        UserWealth::create([
            'user_id' => $user->id,
            'wealth_plan_id' => $plan->id,
            'amount' => $request->amount,
            'start_date' => $start->toDateString(),
            'end_date' => $end->toDateString(),
            'status' => 1,
            'last_earned_at' => $start->toDateString(), // first day counted
        ]);

        return back()->with('success', 'Investment successful!');
    }

    // Update daily earning for an investment
    private function updateDailyEarning(UserWealth $inv, $user)
    {
        $today = Carbon::today();
        $lastEarn = $inv->last_earned_at ? Carbon::parse($inv->last_earned_at) : Carbon::parse($inv->start_date);
        $end = Carbon::parse($inv->end_date);

        // Calculate how many days pending
        $daysPending = $today->diffInDays($lastEarn);
        if ($today->gt($lastEarn) && $today->lte($end)) {
            $dailyProfit = $inv->amount * ($inv->plan->profit_percent / 100) / $inv->plan->days;

            // Add earnings for each pending day
            $totalProfit = $dailyProfit * $daysPending;
            $user->balance += $totalProfit;
            $user->save();

            // Update last_earned_at to today
            $inv->last_earned_at = $today->toDateString();
            $inv->save();
        }

        // If plan ended and still active, return principal + final earnings
        if ($today->gt($end) && $inv->status == 1) {
            $user->balance += $inv->amount; // return principal
            $user->save();

            $inv->status = 0; // completed
            $inv->save();
        }
    }
}
