<?php

namespace App\Services;

use App\Models\User;
use App\Models\Commission;
use Illuminate\Support\Facades\DB;

class CommissionService
{
    /**
     * Distribute commission and credit score for package or task
     *
     * @param int $fromUserId  // user who bought package or completed task
     * @param float $amount    // amount of package or task reward
     * @param string $type     // 'package' or 'task'
     * @param int $maxLevels   // up to 3 levels
     * @param array $meta      // extra data like ['package_id'=>1,'task_id'=>2]
     */
    public function distributeWithCredit(int $fromUserId, float $amount, string $type='package', int $maxLevels=3, array $meta=[])
    {
        $fromUser = User::find($fromUserId);
        if(!$fromUser) return;

        // Load commission rules from DB
        $rules = DB::table('commission_rules')->where('type',$type)->get()->keyBy('level');

        // Load credit score rules from DB
        $scoreRules = DB::table('credit_score_rules')->where('type',$type)->get()->keyBy('level');

        $uplines = $this->getUplines($fromUser,$maxLevels);

        DB::transaction(function() use($uplines,$rules,$scoreRules,$amount,$fromUserId,$type,$meta){
            foreach($uplines as $levelIndex=>$upline){
                $level = $levelIndex+1;

                // 1️⃣ Commission
                if(isset($rules[$level])){
                    $percent = (float)$rules[$level]->percentage;
                    $cAmount = round($amount * $percent / 100, 8);
                    if($cAmount > 0){
                        Commission::create([
                            'user_id' => $upline->id,
                            'from_user_id' => $fromUserId,
                            'type' => $type,
                            'level' => $level,
                            'amount' => $cAmount,
                            'status' => 'approved',
                            'meta' => json_encode($meta)
                        ]);
                        $upline->increment('balance', $cAmount);
                    }
                }

                // 2️⃣ Credit Score
                if(isset($scoreRules[$level])){
                    $score = (int)$scoreRules[$level]->score_points;
                    if($score > 0){
                        $upline->increment('credit_score', $score);
                    }
                }
            }
        });
    }

    /**
     * Get uplines of user up to max levels
     */
    protected function getUplines($user, $levels=3)
    {
        $uplines = [];
        $current = $user;
        for($i=0; $i<$levels; $i++){
            $sponsorId = $current->ref_by ?? $current->ref_id ?? null;
            if(!$sponsorId) break;
            $upline = User::find($sponsorId);
            if(!$upline) break;
            $uplines[] = $upline;
            $current = $upline;
        }
        return $uplines;
    }
}
